<?php

namespace App\Livewire\Contratos;

use Livewire\Component;
use App\Models\Parametro;
use App\Models\Seguimiento;
use Livewire\WithFileUploads;
use App\Models\SeguimientoAdjunto;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class SeguimientoModal extends Component
{
    use WithFileUploads;

    public $modo = 'crear'; // o 'editar'
    public $contrato_id;
    public $proceso_id;
    public $fecha_seguimiento;
    public $estado_seguimiento;
    public $adjuntos = [];
    public $observacion;
    public $seguimiento_id;
    public $adjuntos_existentes = []; 

    protected $listeners = ['abrirModalSeguimiento','editSeguimiento'];

    protected $rules = [
        'adjuntos.*' => 'file|mimes:pdf,jpg,jpeg,png|max:5120',
        'fecha_seguimiento' => 'required|date',
        'estado_seguimiento' => 'required|exists:parametros,id',
    ];

    

    public function abrirModalSeguimiento($contratoId)
    {
        $this->reset(); // Limpia campos anteriores
        $this->contrato_id = $contratoId;
        // Abrir modal desde JS
        // $this->dispatch('abrir-modal', ['id' => 'modalSeguimiento']);
        $this->dispatch('mostrar-modal-seguimiento');
    }

    // public function guardarSeguimiento()
    // {
    //     $this->validate();

    //     $seguimiento = Seguimiento::create([
    //         'contrato_id' => $this->contrato_id,
    //         'observacion' => $this->observacion,
    //         'fecha_seguimiento' => $this->fecha_seguimiento,
    //         'estado_id' => $this->estado_seguimiento,
    //     ]);

    //     foreach ($this->adjuntos as $archivo) {
    //         // Guardamos archivo en storage/app/public/adjuntos
    //         $path = $archivo->store('adjuntos', 'public');

    //         SeguimientoAdjunto::create([
    //             'seguimiento_id' => $seguimiento->id,
    //             'archivo' => $path, // solo el path relativo (ej: adjuntos/archivo.pdf)
    //             'nombre_original' => $archivo->getClientOriginalName(),
    //         ]);
    //     }

    //     if (!empty($this->nuevos_adjuntos)) {
    //         foreach ($this->nuevos_adjuntos as $archivo) {
    //             $path = $archivo->store('adjuntos');

    //             SeguimientoAdjunto::create([
    //                 'seguimiento_id' => $this->seguimiento_id,
    //                 'ruta' => $path,
    //                 'nombre' => $archivo->getClientOriginalName(),
    //             ]);
    //         }
    //     }

    //     $this->reset(['observacion', 'fecha_seguimiento', 'estado_seguimiento', 'adjuntos']);
    //     $this->dispatch('cerrar-modal-seguimiento');
    //     $this->dispatch('seguimiento-guardado');
    //     $this->dispatch('refrescarTabla');
    //     return;
    // }

    public function guardarSeguimiento()
    {
        $this->validate([
            'fecha_seguimiento' => 'required|date',
            'estado_seguimiento' => 'required',
            'adjuntos.*' => 'file|max:10240' // 10MB máximo
        ]);

        if ($this->modo == 'editar') {
            $seguimiento = Seguimiento::find($this->seguimiento_id);
        } else {
            $seguimiento = new Seguimiento();
            $seguimiento->proceso_id = $this->contrato_id;
        }

        $seguimiento->fecha_seguimiento = $this->fecha_seguimiento;
        $seguimiento->estado_id = $this->estado_seguimiento;
        $seguimiento->observacion = $this->observacion;
        $seguimiento->save();

        // Guardar archivos nuevos
        if ($this->adjuntos) {
            foreach ($this->adjuntos as $file) {
                $path = $file->store('adjuntos');
                $seguimiento->archivos()->create([
                    'nombre' => $file->getClientOriginalName(),
                    'ruta' => $path,
                ]);
            }
        }
        $this->dispatch('seguimiento-guardado'); // Puedes cerrar modal y refrescar tabla
        $this->dispatch('refrescarTabla');
    }


    public function render()
    {
        $estados_seguimiento = Parametro::where('tipo', 'estado')->get();
        return view('livewire.contratos.seguimiento-modal',compact('estados_seguimiento'));
    }

    public function editSeguimiento($id)
    {
        $this->modo = 'editar';
        $seguimiento = Seguimiento::with('adjuntos')->findOrFail($id);
        $this->seguimiento_id = $seguimiento->id;
        $this->fecha_seguimiento = $seguimiento->fecha_seguimiento;
        $this->estado_seguimiento = $seguimiento->estado_id;
        $this->observacion = $seguimiento->observacion;
        $this->adjuntos_existentes = $seguimiento->adjuntos->toArray();

        $this->dispatch('abrir-modal'); 
    }

    public function eliminarSeguimiento($id)
    {
        $seguimiento = Seguimiento::findOrFail($id);
        $seguimiento->delete();

        // Eliminar archivos adjuntos si existen
        foreach ($seguimiento->adjuntos as $adjunto) {
            Storage::disk('public')->delete($adjunto->archivo);
            $adjunto->delete();
        }

        $this->dispatch('seguimiento-eliminado');
        $this->dispatch('refrescarTabla');
    }

    public function eliminarAdjunto($id)
    {
        $archivo = SeguimientoAdjunto::find($id);
        if ($archivo) {
            Storage::delete($archivo->archivo);
            $archivo->delete();
            $this->adjuntos_existentes = array_filter($this->adjuntos_existentes, fn($a) => $a['id'] != $id);
        }
    }
}
