<?php

namespace App\Livewire\Contratos;

use Livewire\Component;
use Livewire\Attributes\On;
use App\Models\Proceso as Contrato;
use App\Models\User;

class AsignarTecnicoModal extends Component 
{
    public $contratoId;
    public $responsable_id;
    public $tecnicos = [];
    public $actualResponsableName;
    public $observacion;

    public function mount()
    {
        $this->tecnicos = User::all();
    }

    // SOLUCIÓN DEFINITIVA: Usar listeners tradicionales para evitar el error de unpacking
    protected $listeners = [
        'abrir-modal-asignacion' => 'abrirModal',
        'guardarAsignacion' => 'guardarAsignacion'
    ];

    public function abrirModal($contratoId)
    {
        $this->contratoId = $contratoId;
        $contrato = Contrato::findOrFail($contratoId);
        
        $this->responsable_id = optional($contrato->ultimaAsignacion)->responsable_id;
        $this->actualResponsableName = optional($contrato->ultimaAsignacion?->responsable)->name;
        
        $this->dispatch('mostrar-modal-asignacion');
    
    }

    public function asignar()
    {
        $this->validate([
            'responsable_id' => 'required|exists:users,id',
        ], [
            'responsable_id.required' => 'Debe seleccionar un técnico.',
        ]);

        $contrato = Contrato::findOrFail($this->contratoId);
        $actual = optional($contrato->ultimaAsignacion)->responsable_id;

        if ($this->responsable_id == $actual) {
            $this->dispatch('asignacion-exitosa', ['mensaje' => 'El técnico ya está asignado.']);
            $this->dispatch('cerrar-modal-asignacion');
            return;
        }

        // Si hay técnico actual y se quiere cambiar, pedir confirmación
        if ($actual) {
            $nombreActual = optional($contrato->ultimaAsignacion?->responsable)->name;
            $nombreNuevo  = optional(User::find($this->responsable_id))->name;

            $this->dispatch('confirmar-reemplazo', [
                'actual' => $nombreActual,
                'nuevo'  => $nombreNuevo,
            ]);
            return;
        }

        // Si no hay técnico actual, guardar directamente
        $this->guardarAsignacion();
    }

    // También usar listeners tradicionales para este método

    public function guardarAsignacion()
    {
        $contrato = Contrato::findOrFail($this->contratoId);
        $actual = optional($contrato->ultimaAsignacion)->responsable_id;

        if ($actual) {
            $contrato->asignaciones()->detach($actual);
        }

        $contrato->asignaciones()->attach($this->responsable_id, [
            'fecha_asignacion' => now(),
            'observacion' => $this->observacion,
        ]);

        $this->dispatch('asignacion-exitosa', ['mensaje' => 'Técnico asignado correctamente.']);
        $this->dispatch('cerrar-modal-asignacion');
        $this->dispatch('refrescarTabla');
    }

    public function cerrarModalAsignacion(){
        $this->reset(['contratoId', 'responsable_id', 'observacion']);
        $this->dispatch('cerrar-modal-asignacion');
    }

    public function render()
    {
        return view('livewire.contratos.asignar-tecnico-modal');
    }
}